<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Channel;

class ChannelController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $channels = Channel::where('status', 1)->get(); // Fetch active channel
        $trashedChannels = channel::onlyTrashed()->get(); // Fetch soft-deleted channel
        return view('admin.channels.index', compact('channels', 'trashedChannels'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.channels.create');
    }

    /**
     * Store a newly created channel in the database.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255', 'unique:channels,name'],
        ]);

        channel::create($request->all());

        return redirect()->route('admin.channels.index')->with('success', 'Channel created successfully.');
    }

    /**
     * Display the specified channel.
     */
    public function show(Channel $channel)
    {
        return view('admin.channels.show', compact('channel'));
    }

    /**
     * Show the form for editing the specified channel.
     */
    public function edit(Channel $channel)
    {
        return view('admin.channels.edit', compact('channel'));
    }

    /**
     * Update the specified channel in the database.
     */
    public function update(Request $request, Channel $channel)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255', 'unique:Channels,name,' . $channel->id],
        ]);

        $channel->update($request->all());

        return redirect()->route('admin.channels.index')->with('success', 'Channel updated successfully.');
    }

    /**
     * Soft delete the specified channel.
     */
    public function destroy(Channel $channel)
    {
        $channel->delete(); // Soft delete
        return redirect()->route('admin.channels.index')->with('success', 'Channel soft deleted successfully.');
    }

    /**
     * Restore a soft-deleted channel.
     */
    public function restore($id)
    {
        $channel = Channel::withTrashed()->find($id);
        if ($channel) {
            $channel->restore();
            return redirect()->route('admin.channels.index')->with('success', 'Channel restored successfully.');
        }
        return redirect()->route('admin.channels.index')->with('error', 'Channel not found.');
    }

    public function toggleStatus(Channel $channel)
    {
        $channel->status = !$channel->status;
        $channel->save();

        return redirect()->route('admin.channels.index')->with('success', 'Channel status updated successfully.');
    }

    public function enabledChannels(){
        $channels = Channel::where('status', 1)->get();
        return view('admin.channels.index', compact('channels'));
    }

    public function disabledChannels(){
        $channels = Channel::where('status', 0)->get();
        return view('admin.channels.index', compact('channels'));
    }

    /**
     * Permanently delete a soft-deleted channel.
     */
    public function forceDelete($id)
    {
        $channel = Channel::withTrashed()->find($id);
        if ($channel) {
            $channel->forceDelete(); // Hard delete
            return redirect()->route('admin.channels.index')->with('success', 'Channel permanently deleted.');
        }
        return redirect()->route('admin.channels.index')->with('error', 'Channel not found.');
    }

    public function trashed()
    {
        $channels = Channel::onlyTrashed()->get();
        return view('admin.channels.trashed', compact('channels'));
    }
}
