<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Mail;
use App\Mail\UserCreated;

class UserController extends Controller
{
    public function index()
    {
        $users = User::where('status', 1)->with('roles')->get();
        return view('admin.users.index', compact('users'));
    }

    public function create()
    {
        $roles = Role::all();
        return view('admin.users.create', compact('roles'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users',
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string|max:500',
            'user_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:10240',
        ]);

        $imagePath = null;
        if ($request->hasFile('user_image')) {
            $imagePath = $request->file('user_image')->store('uploads/users', 'public');
        }

        $randomPassword = Str::random(8);

        $user = User::create([
            'first_name' => $request->first_name,
            'last_name' => $request->last_name,
            'name' => $request->name,
            'email' => $request->email,
            'line_name' => $request->line_name,
            // 'phone' => $request->phone,
            'address' => $request->address,
            'other_address' => $request->other_address,
            'bank_info' => $request->bank_info,
            'other' => $request->other,
            'started_on' => $request->started_on,
            'line_group_chat' => $request->line_group_chat,
            'personal_email' => $request->personal_email,
            'work_email' => json_encode($request->work_email),
            'share_email' => json_encode($request->share_email),
            'gender' => $request->gender,
            'age' => $request->age,
            'birth_date' => $request->birth_date,
            'residence' => $request->residence,
            'working_hours' => $request->working_hours,
            'working_hours_other' => $request->working_hours_other,
            'operating_location' => $request->operating_location,
            'detail' => $request->detail,
            'user_image' => $imagePath,
            'password' => Hash::make($randomPassword),
            'phone_number' => json_encode($request->phone_number),
            'phone_number_type' => json_encode($request->phone_number_type),
            'local_number' => json_encode($request->local_number),
            'local_number_type' => json_encode($request->local_number_type),
        ]);

        $role = Role::find($request->role_id);
        if ($role) {
            $user->assignRole($role->name);
        }

        Mail::to($user->email)->send(new UserCreated($user, $randomPassword));

        return to_route('admin.users.index')->with('success', 'User created successfully and email sent.');
    }

    public function show(User $user)
    {
        $roles = $user->roles;
        return view('admin.users.show', compact('user', 'roles'));
    }

    public function edit(User $user)
    {
        $roles = Role::all();
        $assignedRoles = $user->roles->pluck('id')->toArray();
        return view('admin.users.edit', compact('user', 'roles', 'assignedRoles'));
    }

    public function update(Request $request, User $user)
    {
        $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string|max:500',
            'role_id' => 'required|exists:roles,id',
        ]);

        $imagePath = $user->user_image;
        if ($request->hasFile('user_image')) {
            $imagePath = $request->file('user_image')->store('uploads/users', 'public');
        }

        $user->update([
            'first_name' => $request->first_name,
            'last_name' => $request->last_name,
            'name' => $request->name,
            'email' => $request->email,
            'line_name' => $request->line_name,
            // 'phone' => $request->phone,
            'address' => $request->address,
            'other_address' => $request->other_address,
            'bank_info' => $request->bank_info,
            'other' => $request->other,
            'started_on' => $request->started_on,
            'line_group_chat' => $request->line_group_chat,
            'personal_email' => $request->personal_email,
            'work_email' => json_encode($request->work_email),
            'share_email' => json_encode($request->share_email),
            'gender' => $request->gender,
            'age' => $request->age,
            'birth_date' => $request->birth_date,
            'residence' => $request->residence,
            'working_hours' => $request->working_hours,
            'working_hours_other' => $request->working_hours_other,
            'operating_location' => $request->operating_location,
            'detail' => $request->detail,
            'user_image' => $imagePath,
            'phone_number' => json_encode($request->phone_number),
            'phone_number_type' => json_encode($request->phone_number_type),
            'local_number' => json_encode($request->local_number),
            'local_number_type' => json_encode($request->local_number_type),
            ]);

        $user->roles()->detach();
        $role = Role::find($request->role_id);
        if ($role) {
            $user->assignRole($role->name);
        }

        return redirect()->route('admin.users.index')->with('success', 'User updated successfully.');
    }


    public function destroy(User $user)
    {
        $user->delete();
        return redirect()->route('admin.users.index')->with('success', 'User deleted successfully.');
    }

    public function toggleStatus(User $user)
    {
        $user->status = !$user->status;
        $user->save();

        return redirect()->route('admin.users.index')->with('success', 'User status updated successfully.');
    }

    public function enabledUsers(){
        $users = User::where('status', 1)->with('roles')->get();
        return view('admin.users.index', compact('users'));
    }

    public function disabledUsers(){
        $users = User::where('status', 0)->with('roles')->get();
        return view('admin.users.index', compact('users'));
    }

    public function trashed()
    {
        $users = User::onlyTrashed()->get();
        return view('admin.users.trashed', compact('users'));
    }
}
