<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Channel;
use App\Models\Tour;
use App\Models\Region;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Auth;
use App\Models\Group;
use App\Models\User;
use App\Models\TourType;

class TourSettingController extends Controller
{
     /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $tours = Tour::where('status', 1)->with(['tourTypes', 'city'])->paginate(10);
        return view('admin.tour_setting.index', compact('tours'));
    }

    public function importTour(Request $request)
    {
        $request->validate([
            'bokun_id' => 'required|numeric'
        ]);

        $bokunId = $request->input('bokun_id');
        if (Tour::where('bokun_id', $bokunId)->exists()) {
            return back()->with('error', 'This Bokun ID already exists in the database.');
        }

        $accessKey = "3ba2e588cfb94197aa8addfafa67389d";
        $secretKey = "bfb3e1e0433c491891543f9b43cf710a";
        $apiPath = "/activity.json/{$bokunId}?currency=ISK&lang=EN";
        $apiUrl = "https://api.bokun.io" . $apiPath;

        // Generate HMAC Signature
        $bokunDate = gmdate("Y-m-d H:i:s"); // Current UTC time
        $stringToSign = "{$bokunDate}{$accessKey}GET{$apiPath}";
        $signature = base64_encode(hash_hmac("sha1", $stringToSign, $secretKey, true));

        // API Call
        $response = Http::withHeaders([
            'X-Bokun-Date' => $bokunDate,
            'X-Bokun-AccessKey' => $accessKey,
            'X-Bokun-Signature' => $signature,
            'Accept' => 'application/json',
        ])->get($apiUrl);

        if ($response->failed()) {
            return back()->with('error', 'Failed to fetch tour details. Please check the Bokun ID.');
        }

        $tourData = $response->json();
        if (!isset($tourData['id']) || !isset($tourData['title'])) {
            return back()->with('error', 'Invalid Bokun ID. No tour found.');
        }

        Tour::create([
            'uid' => Auth::user()->id,
            'bokun_id' => $tourData['id'],
            'bokun_title' => $tourData['title']
        ]);

        return back()->with('success', 'Tour imported successfully!');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
      //
    }

    /**
     * Store in the database.
     */
    public function store(Request $request)
    {
      //
    }

    /**
     * Display the specified tour.
     */
    public function show($tour)
    {
        $tour = Tour::with(['tourTypes', 'managers', 'region', 'prefecture', 'city'])->findOrFail($tour);
        return view('admin.tour_setting.show', compact('tour'));
    }

    /**
     * Show the form for editing the specified tour.
     */
    public function edit($tour)
    {
        $tour = Tour::with(['tourTypes', 'managers', 'region', 'prefecture', 'city'])->findOrFail($tour);
        $regions = Region::all();
        $groups = Group::where('status', 1)->get();
        $managers = User::role('manager')->get();
        $tourTypes = TourType::where('tour_id', $tour->id)->get();
        return view('admin.tour_setting.edit', compact('tour', 'regions', 'groups', 'managers', 'tourTypes'));
    }

    /**
     * Update the specified channel in the database.
     */
    public function update(Request $request, $tour)
    {
        // dd($request->all());
        $tour = Tour::findOrFail($tour);
       // dd($tour);
        // Validation
        $request->validate([
            'name' => 'nullable|string|max:255',
            'time' => 'nullable|string',
            'start_time' => 'nullable|date_format:H:i',
            'end_time' => 'nullable|date_format:H:i',
            'duration' => 'nullable|integer',
            'region_id' => 'nullable|exists:regions,id',
            'prefecture_id' => 'nullable|exists:prefectures,id',
            'city_id' => 'nullable|exists:cities,id',
            'operation_area' => 'nullable|string',
            'operation_lat_lng' => 'nullable|string',
            'guide_free' => 'nullable|string',
            'manager' => 'nullable|array',
            'manager.*' => 'exists:users,id',
            'link_form' => 'nullable|url',
            'tour_instructions' => 'nullable|string',
            'tour_type' => 'nullable|array',
            'tour_type.*' => 'nullable|string',
            'rate' => 'nullable|array',
            'rate.*' => 'nullable|numeric',
            'meet_point' => 'nullable|array',
            'meet_lat_lng' => 'nullable|array',
            'response_template' => 'nullable|array',
            'response_template.*' => 'exists:groups,id',
        ]);

        // dd($request->all());
        // Update Basic Fields
        $tour->update([
            'name' => $request->name,
            'time' => $request->time,
            'start_time' => $request->start_time,
            'end_time' => $request->end_time,
            'duration' => $request->duration,
            'region_id' => $request->region_id,
            'prefecture_id' => $request->prefecture_id,
            'city_id' => $request->city_id,
            'operation_area' => $request->operation_area,
            'operation_lat_lng' => $request->operation_lat_lng,
            'guide_free' => $request->guide_free,
            'link_form' => $request->link_form,
            'tour_instructions' => $request->tour_instructions,
        ]);

        // Managers Update
        if ($request->has('manager')) {
            $tour->managers()->sync($request->manager);
        }

        // Tour Types Update
        if ($request->has('tour_type')) {
            $tourTypes = [];
            foreach ($request->tour_type as $index => $tourType) {
                if (!empty($tourType)) {
                    $tourTypes[] = [
                        'name' => $tourType,
                        'rate' => $request->rate[$index] ?? 0,
                        'tour_id' => $tour->id,
                    ];
                }
            }
            TourType::where('tour_id', $tour->id)->delete();
            TourType::insert($tourTypes);
        }

        // Meet Points Update
        if ($request->has('meet_point')) {
            $tour->meet_point = json_encode($request->meet_point);
            $tour->meet_lat_lng = json_encode($request->meet_lat_lng);
            $tour->save();
        }

        // Response Template Update
        if ($request->has('response_template')) {
            $tour->groups()->sync($request->response_template);
        }

        return redirect()->route('admin.tour_settings.index')->with('success', 'Tour updated successfully!');
    }


    /**
     * Soft delete the specified channel.
     */
    public function destroy($tour)
    {
        $tour = Tour::find($tour);
        $tour->delete(); // Soft delete
        return redirect()->route('admin.tour_settings.index')->with('success', 'Tour soft deleted successfully.');
    }

    /**
     * Restore a soft-deleted channel.
     */
    public function restore($id)
    {
        $tour = Tour::withTrashed()->find($id);
        if ($tour) {
            $tour->restore();
            return redirect()->route('admin.tour_settings.index')->with('success', 'Tour restored successfully.');
        }
        return redirect()->route('admin.tour_settings.index')->with('error', 'Tour not found.');
    }

    public function toggleStatus($tour)
    {
        $tour = Tour::find($tour);
        $tour->status = !$tour->status;
        $tour->save();

        return redirect()->route('admin.tour_settings.index')->with('success', 'Tour status updated successfully.');
    }

    public function publishedTours(){
        $tours = Tour::where('status', 1)->paginate(10);
        return view('admin.tour_setting.index', compact('tours'));
    }

    public function unpublishedTours(){
        $tours = Tour::where('status', 0)->paginate(10);
        return view('admin.tour_setting.index', compact('tours'));
    }

    /**
     * Permanently delete a soft-deleted channel.
     */
    public function forceDelete($id)
    {
        $tour = Tour::withTrashed()->find($id);
        if ($tour) {
            $tour->forceDelete(); // Hard delete
            return redirect()->route('admin.tour_settings.index')->with('success', 'Tour permanently deleted.');
        }
        return redirect()->route('admin.tour_settings.index')->with('error', 'Tour not found.');
    }

    public function trashed()
    {
        $tours = Tour::onlyTrashed()->get();
        return view('admin.tour_setting.trashed', compact('tours'));
    }
}
