<?php
namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Group;
use Illuminate\Http\Request;

class GroupController extends Controller
{
    /**
     * Display a listing of active and deleted groups.
     */
    public function index()
    {
        $groups = Group::where('status', 1)->get(); // Fetch active groups
        $trashedGroups = Group::onlyTrashed()->get(); // Fetch soft-deleted groups
        return view('admin.groups.index', compact('groups', 'trashedGroups'));
    }

    /**
     * Show the form for creating a new group.
     */
    public function create()
    {
        return view('admin.groups.create');
    }

    /**
     * Store a newly created group in the database.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255', 'unique:groups,name'],
        ]);

        Group::create($request->all());

        return redirect()->route('admin.groups.index')->with('success', 'Group created successfully.');
    }

    /**
     * Display the specified group.
     */
    public function show(Group $group)
    {
        return view('admin.groups.show', compact('group'));
    }

    /**
     * Show the form for editing the specified group.
     */
    public function edit(Group $group)
    {
        return view('admin.groups.edit', compact('group'));
    }

    /**
     * Update the specified group in the database.
     */
    public function update(Request $request, Group $group)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255', 'unique:groups,name,' . $group->id],
        ]);

        $group->update($request->all());

        return redirect()->route('admin.groups.index')->with('success', 'Group updated successfully.');
    }

    /**
     * Soft delete the specified group.
     */
    public function destroy(Group $group)
    {
        $group->delete(); // Soft delete
        return redirect()->route('admin.groups.index')->with('success', 'Group soft deleted successfully.');
    }

    /**
     * Restore a soft-deleted group.
     */
    public function restore($id)
    {
        $group = Group::withTrashed()->find($id);
        if ($group) {
            $group->restore();
            return redirect()->route('admin.groups.index')->with('success', 'Group restored successfully.');
        }
        return redirect()->route('admin.groups.index')->with('error', 'Group not found.');
    }

    public function toggleStatus(Group $group)
    {
        $group->status = !$group->status;
        $group->save();

        $responseTemplates = $group->responseTemplates;

        foreach ($responseTemplates as $responseTemplate) {
            $responseTemplate->status = $group->status;
            $responseTemplate->save();
        }

        return redirect()->route('admin.groups.index')->with('success', 'Group status updated successfully.');
    }

    public function enabledGroups(){
        $groups = Group::where('status', 1)->get();
        return view('admin.groups.index', compact('groups'));
    }

    public function disabledGroups(){
        $groups = Group::where('status', 0)->get();
        return view('admin.groups.index', compact('groups'));
    }

    /**
     * Permanently delete a soft-deleted group.
     */
    public function forceDelete($id)
    {
        $group = Group::withTrashed()->find($id);
        if ($group) {
            $group->forceDelete(); // Hard delete
            return redirect()->route('admin.groups.index')->with('success', 'Group permanently deleted.');
        }
        return redirect()->route('admin.groups.index')->with('error', 'Group not found.');
    }

    public function trashed()
    {
        $trashedGroups = Group::onlyTrashed()->get();
        return view('admin.groups.trashed', compact('trashedGroups'));
    }

}
